<?php
/**
 * Google Apps Script Generator for WooCommerce to Google Sheets Integration
 */

namespace UGSIW;

if (!defined('ABSPATH')) {
    exit;
}

class UGSIW_Script_Generator {
    
    /**
     * Available fields for reference
     */
    private $available_fields;
    
    /**
     * Constructor
     */
    public function __construct($available_fields) {
        $this->available_fields = $available_fields;
    }
    
    /**
     * Generate Google Apps Script based on mode and Pro features
     */
    public function generate_script($selected_fields, $monthly_sheets = false, $pro_features = array()) {
        // Check if Pro features are enabled
        $daily_weekly = isset($pro_features['daily_weekly']) ? $pro_features['daily_weekly'] : 'none';
        $product_sheets = isset($pro_features['product_sheets']) ? $pro_features['product_sheets'] : false;
        $custom_sheet_name = isset($pro_features['custom_sheet_name']) ? $pro_features['custom_sheet_name'] : false;
        $custom_template = isset($pro_features['custom_template']) ? $pro_features['custom_template'] : '';
        
        // Determine which script to generate
        if ($product_sheets) {
            return $this->generate_product_sheet_script($selected_fields, $custom_sheet_name, $custom_template);
        } elseif ($daily_weekly === 'daily') {
            return $this->generate_daily_sheet_script($selected_fields, $custom_sheet_name, $custom_template);
        } elseif ($daily_weekly === 'weekly') {
            return $this->generate_weekly_sheet_script($selected_fields, $custom_sheet_name, $custom_template);
        } elseif ($monthly_sheets) {
            return $this->generate_monthly_sheet_script($selected_fields, $custom_sheet_name, $custom_template);
        } else {
            return $this->generate_single_sheet_script($selected_fields, $custom_sheet_name, $custom_template);
        }
    }
    
    /**
     * Generate Google Apps Script code for single sheet - PRO VERSION
     */
    private function generate_single_sheet_script($selected_fields, $custom_sheet_name = false, $custom_template = '') {
        // Get field labels for headers
        $headers = array();
        $field_mapping = array();

        foreach ($selected_fields as $field_key) {
            if (isset($this->available_fields[$field_key])) {
                $headers[] = $this->available_fields[$field_key]['label'];
                $field_mapping[$field_key] = $this->available_fields[$field_key]['label'];
            }
        }

        $headers_js = json_encode($headers, JSON_PRETTY_PRINT);
        $field_mapping_js = json_encode($field_mapping, JSON_PRETTY_PRINT);
        $field_list = esc_js($this->get_field_list($selected_fields));

        $script  = "// Google Apps Script Code for Google Sheets\n";
        $script .= "// Generated by WP Methods WooCommerce to Google Sheets Plugin\n";
        $script .= "// Single Sheet Mode\n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "// Custom Sheet Name: {$custom_template}\n";
        }
        $script .= "// Fields: {$field_list}\n\n";

        // Add custom sheet name functionality if enabled
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= $this->get_custom_sheet_name_functions($custom_template);
        }

        $script .= "function doPost(e) {\n";
        $script .= "  try {\n";
        $script .= "    const data = JSON.parse(e.postData.contents);\n";
        
        // Get sheet based on custom naming or default
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "    const sheetName = generateCustomSheetName(data);\n";
            $script .= "    const sheet = getOrCreateSheet(sheetName);\n";
        } else {
            $script .= "    const sheet = SpreadsheetApp.getActiveSpreadsheet().getActiveSheet();\n";
        }
        
        $script .= "    \n";
        $script .= "    // Initialize sheet if needed with 'Last Updated' column\n";
        $script .= "    if (sheet.getLastRow() === 0) {\n";
        $script .= "      const headers = {$headers_js};\n";
        $script .= "      // Add 'Last Updated' column from the start\n";
        $script .= "      headers.push('Last Updated');\n";
        $script .= "      sheet.appendRow(headers);\n";
        $script .= "      const headerRange = sheet.getRange(1, 1, 1, headers.length);\n";
        $script .= "      headerRange.setBackground('#4CAF50').setFontColor('white').setFontWeight('bold');\n";
        $script .= "      // Format the Last Updated header\n";
        $script .= "      sheet.getRange(1, headers.length).setBackground('#FFC107').setFontWeight('bold');\n";
        $script .= "      sheet.setFrozenRows(1);\n";
        $script .= "      console.log('Sheet initialized with headers:', headers);\n";
        $script .= "    }\n\n";


        $script .= "// Function to format date columns\n";
        $script .= "function formatDateColumns(sheet, headers) {\n";
        $script .= "  const orderDateIndex = headers.indexOf('Order Date');\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  \n";
        $script .= "  if (orderDateIndex !== -1) {\n";
        $script .= "    const orderDateColumn = orderDateIndex + 1;\n";
        $script .= "    const range = sheet.getRange(2, orderDateColumn, sheet.getLastRow() - 1, 1);\n";
        $script .= "    // Set number format to date/time\n";
        $script .= "    range.setNumberFormat('yyyy-mm-dd hh:mm:ss');\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    const lastUpdatedColumn = lastUpdatedIndex + 1;\n";
        $script .= "    const range = sheet.getRange(2, lastUpdatedColumn, sheet.getLastRow() - 1, 1);\n";
        $script .= "    // Set number format to date/time\n";
        $script .= "    range.setNumberFormat('yyyy-mm-dd hh:mm:ss');\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "    // Find order ID column index\n";
        $script .= "    const headers = sheet.getRange(1, 1, 1, sheet.getLastColumn()).getValues()[0];\n";
        $script .= "    const orderIdColumn = headers.indexOf('Order ID') + 1;\n";
        $script .= "    \n";
        $script .= "    if (orderIdColumn === 0) {\n";
        $script .= "      throw new Error('Order ID column not found in sheet headers');\n";
        $script .= "    }\n\n";

        $script .= "    // Check if order already exists\n";
        $script .= "    let existingRow = null;\n";
        $script .= "    if (sheet.getLastRow() > 1) {\n";
        $script .= "      const orderIds = sheet.getRange(2, orderIdColumn, sheet.getLastRow() - 1, 1).getValues();\n";
        $script .= "      \n";
        $script .= "      for (let i = 0; i < orderIds.length; i++) {\n";
        $script .= "        const rowOrderId = orderIds[i][0];\n";
        $script .= "        if (rowOrderId && rowOrderId.toString() === data.order_id.toString()) {\n";
        $script .= "          existingRow = i + 2; // +2 because headers and 0-based index\n";
        $script .= "          break;\n";
        $script .= "        }\n";
        $script .= "      }\n";
        $script .= "    }\n\n";

        $script .= "    if (existingRow) {\n";
        $script .= "      // Update existing order\n";
        $script .= "      updateOrderRow(sheet, existingRow, data, headers);\n";
        $script .= "      \n";
        $script .= "      return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "        status: 'success',\n";
        $script .= "        message: 'Order updated successfully',\n";
        $script .= "        order_id: data.order_id,\n";
        $script .= "        row: existingRow\n";
        $script .= "      })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "    } else {\n";
        $script .= "      // Add new order\n";
        $script .= "      const newRow = addNewOrderRow(sheet, data, headers);\n";
        $script .= "      \n";
        $script .= "      return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "        status: 'success',\n";
        $script .= "        message: 'Order added successfully',\n";
        $script .= "        order_id: data.order_id,\n";
        $script .= "        row: newRow\n";
        $script .= "      })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "    }\n";

        $script .= "  } catch (error) {\n";
        $script .= "    console.error('Error:', error);\n";
        $script .= "    return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "      status: 'error',\n";
        $script .= "      message: error.toString(),\n";
        $script .= "      stack: error.stack\n";
        $script .= "    })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "function updateOrderRow(sheet, row, data, headers) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  \n";
        $script .= "  // Update each field that exists in the data\n";
        $script .= "  for (const key in data) {\n";
        $script .= "    if (fieldMap[key]) {\n";
        $script .= "      // Find which column this field is in\n";
        $script .= "      const columnIndex = headers.indexOf(fieldMap[key]) + 1;\n";
        $script .= "      if (columnIndex > 0) {\n";
        $script .= "        sheet.getRange(row, columnIndex).setValue(data[key]);\n";
        $script .= "      }\n";
        $script .= "    }\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  // Find 'Last Updated' column\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    // Set the timestamp in existing 'Last Updated' column\n";
        $script .= "    sheet.getRange(row, lastUpdatedIndex + 1).setValue(new Date());\n";
        $script .= "  } else {\n";
        $script .= "    console.error('Last Updated column not found in headers:', headers);\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  console.log('Updated order', data.order_id, 'in row', row);\n";
        $script .= "}\n\n";

        $script .= "function addNewOrderRow(sheet, data, headers) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  const rowData = [];\n";
        $script .= "  \n";
        $script .= "  // Build row in the correct order based on headers\n";
        $script .= "  headers.forEach(header => {\n";
        $script .= "    let found = false;\n";
        $script .= "    for (const key in fieldMap) {\n";
        $script .= "      if (fieldMap[key] === header) {\n";
        $script .= "        rowData.push(data[key] || '');\n";
        $script .= "        found = true;\n";
        $script .= "        break;\n";
        $script .= "      }\n";
        $script .= "    }\n";
        $script .= "    if (!found && header !== 'Last Updated') {\n";
        $script .= "      rowData.push('');\n";
        $script .= "    }\n";
        $script .= "  });\n";
        $script .= "  \n";
        $script .= "  // Add timestamp for Last Updated column\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    // Insert timestamp at the correct position\n";
        $script .= "    rowData[lastUpdatedIndex] = new Date();\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  // Add the row\n";
        $script .= "  sheet.appendRow(rowData);\n";
        $script .= "  const newRow = sheet.getLastRow();\n";
        $script .= "  \n";
        $script .= "  // Format date columns\n";
        $script .= "  formatDateColumns(sheet, headers);\n";
        $script .= "  \n";
        $script .= "  console.log('Added new order', data.order_id, 'in row', newRow);\n";
        $script .= "  return newRow;\n";
        $script .= "}\n\n";

        $script .= "function getFieldKeyFromLabel(label) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  for (const key in fieldMap) {\n";
        $script .= "    if (fieldMap[key] === label) return key;\n";
        $script .= "  }\n";
        $script .= "  return label.toLowerCase().replace(/ /g, '_');\n";
        $script .= "}\n\n";

        $script .= "// Test function - run this manually to see if the script works\n";
        $script .= "function testUpdate() {\n";
        $script .= "  const testData = {\n";
        $script .= "    order_id: 12345,\n";
        $script .= "    billing_name: 'Test Customer',\n";
        $script .= "    billing_email: 'test@example.com',\n";
        $script .= "    order_status: 'processing',\n";
        $script .= "    order_amount_with_currency: '$100.00',\n";
        $script .= "    order_date: '2024-01-15 10:30:00'\n";
        $script .= "  };\n";
        $script .= "  \n";
        $script .= "  // Simulate a POST request\n";
        $script .= "  const mockPost = {\n";
        $script .= "    postData: {\n";
        $script .= "      contents: JSON.stringify(testData)\n";
        $script .= "    }\n";
        $script .= "  };\n";
        $script .= "  \n";
        $script .= "  const result = doPost(mockPost);\n";
        $script .= "  console.log('Test result:', result.getContent());\n";
        $script .= "}\n\n";

        $script .= "// Force update all orders - useful if you need to re-sync\n";
        $script .= "function forceInitialize() {\n";
        $script .= "  const sheet = SpreadsheetApp.getActiveSpreadsheet().getActiveSheet();\n";
        $script .= "  sheet.clear();\n";
        $script .= "  \n";
        $script .= "  const headers = {$headers_js};\n";
        $script .= "  // Add 'Last Updated' column from the start\n";
        $script .= "  headers.push('Last Updated');\n";
        $script .= "  sheet.appendRow(headers);\n";
        $script .= "  const headerRange = sheet.getRange(1, 1, 1, headers.length);\n";
        $script .= "  headerRange.setBackground('#4CAF50').setFontColor('white').setFontWeight('bold');\n";
        $script .= "  // Format the Last Updated header\n";
        $script .= "  sheet.getRange(1, headers.length).setBackground('#FFC107').setFontWeight('bold');\n";
        $script .= "  sheet.setFrozenRows(1);\n";
        $script .= "  \n";
        $script .= "  console.log('Sheet initialized with headers:', headers);\n";
        $script .= "}\n";

        return $script;
    }


    /**
     * Generate Google Apps Script code for daily sheets - PRO FEATURE
     */
    private function generate_daily_sheet_script($selected_fields, $custom_sheet_name = false, $custom_template = '') {
        $headers = array();
        $field_mapping = array();

        foreach ($selected_fields as $field_key) {
            if (isset($this->available_fields[$field_key])) {
                $headers[] = $this->available_fields[$field_key]['label'];
                $field_mapping[$field_key] = $this->available_fields[$field_key]['label'];
            }
        }

        $headers_js = json_encode($headers, JSON_PRETTY_PRINT);
        $field_mapping_js = json_encode($field_mapping, JSON_PRETTY_PRINT);
        $field_list = esc_js($this->get_field_list($selected_fields));

        $script  = "// Google Apps Script Code for Google Sheets\n";
        $script .= "// Generated by WP Methods WooCommerce to Google Sheets Plugin - PRO VERSION\n";
        $script .= "// Daily Sheets Mode\n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "// Custom Sheet Name: {$custom_template}\n";
        }
        $script .= "// Fields: {$field_list}\n\n";

        // Add helper functions
        $script .= "function getOrCreateSheet(sheetName) {\n";
        $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
        $script .= "  let sheet = ss.getSheetByName(sheetName);\n\n";
        $script .= "  if (!sheet) {\n";
        $script .= "    sheet = ss.insertSheet(sheetName);\n";
        $script .= "    const headers = {$headers_js};\n";
        $script .= "    // Add 'Last Updated' column from the start\n";
        $script .= "    headers.push('Last Updated');\n";
        $script .= "    sheet.appendRow(headers);\n";
        $script .= "    const headerRange = sheet.getRange(1, 1, 1, headers.length);\n";
        $script .= "    headerRange.setBackground('#4CAF50').setFontColor('white').setFontWeight('bold');\n";
        $script .= "    // Format the Last Updated header\n";
        $script .= "    sheet.getRange(1, headers.length).setBackground('#FFC107').setFontWeight('bold');\n";
        $script .= "    sheet.setFrozenRows(1);\n";
        $script .= "    console.log('Created new daily sheet:', sheetName);\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  return sheet;\n";
        $script .= "}\n\n";

        $script .= "function formatDateColumns(sheet, headers) {\n";
        $script .= "  const orderDateIndex = headers.indexOf('Order Date');\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  \n";
        $script .= "  if (orderDateIndex !== -1) {\n";
        $script .= "    const orderDateColumn = orderDateIndex + 1;\n";
        $script .= "    const range = sheet.getRange(2, orderDateColumn, sheet.getLastRow() - 1, 1);\n";
        $script .= "    range.setNumberFormat('yyyy-mm-dd hh:mm:ss');\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    const lastUpdatedColumn = lastUpdatedIndex + 1;\n";
        $script .= "    const range = sheet.getRange(2, lastUpdatedColumn, sheet.getLastRow() - 1, 1);\n";
        $script .= "    range.setNumberFormat('yyyy-mm-dd hh:mm:ss');\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "function getDailySheetName(dateString) {\n";
        $script .= "  try {\n";
        $script .= "    let date;\n";
        $script .= "    if (dateString.includes('T')) {\n";
        $script .= "      date = new Date(dateString);\n";
        $script .= "    } else if (dateString.includes('-')) {\n";
        $script .= "      const parts = dateString.split(' ')[0].split('-');\n";
        $script .= "      date = new Date(parts[0], parts[1] - 1, parts[2] || 1);\n";
        $script .= "    } else {\n";
        $script .= "      date = new Date();\n";
        $script .= "    }\n";
        $script .= "    \n";
        $script .= "    const year = date.getFullYear();\n";
        $script .= "    const month = String(date.getMonth() + 1).padStart(2, '0');\n";
        $script .= "    const day = String(date.getDate()).padStart(2, '0');\n";
        $script .= "    \n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "    // Use custom template\n";
            $script .= "    let sheetName = '{$custom_template}';\n";
            $script .= "    sheetName = sheetName.replace(/{year}/g, year);\n";
            $script .= "    sheetName = sheetName.replace(/{month}/g, month);\n";
            $script .= "    sheetName = sheetName.replace(/{day}/g, day);\n";
            $script .= "    sheetName = sheetName.replace(/{site_name}/g, 'Daily Orders');\n";
            $script .= "    return sheetName;\n";
        } else {
            $script .= "    return `Daily - \${year}-\${month}-\${day}`;\n";
        }
        $script .= "  } catch (e) {\n";
        $script .= "    const now = new Date();\n";
        $script .= "    const year = now.getFullYear();\n";
        $script .= "    const month = String(now.getMonth() + 1).padStart(2, '0');\n";
        $script .= "    const day = String(now.getDate()).padStart(2, '0');\n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "    let sheetName = '{$custom_template}';\n";
            $script .= "    sheetName = sheetName.replace(/{year}/g, year);\n";
            $script .= "    sheetName = sheetName.replace(/{month}/g, month);\n";
            $script .= "    sheetName = sheetName.replace(/{day}/g, day);\n";
            $script .= "    sheetName = sheetName.replace(/{site_name}/g, 'Daily Orders');\n";
            $script .= "    return sheetName;\n";
        } else {
            $script .= "    return `Daily - \${year}-\${month}-\${day}`;\n";
        }
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "function updateOrderRow(sheet, row, data, headers) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  \n";
        $script .= "  // Update each field that exists in the data\n";
        $script .= "  for (const key in data) {\n";
        $script .= "    if (fieldMap[key]) {\n";
        $script .= "      const columnIndex = headers.indexOf(fieldMap[key]) + 1;\n";
        $script .= "      if (columnIndex > 0) {\n";
        $script .= "        sheet.getRange(row, columnIndex).setValue(data[key]);\n";
        $script .= "      }\n";
        $script .= "    }\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  // Find 'Last Updated' column\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    const now = new Date();\n";
        $script .= "    const formattedDate = Utilities.formatDate(now, Session.getScriptTimeZone(), 'yyyy-MM-dd HH:mm:ss');\n";
        $script .= "    sheet.getRange(row, lastUpdatedIndex + 1).setValue(formattedDate);\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  formatDateColumns(sheet, headers);\n";
        $script .= "  \n";
        $script .= "  console.log('Updated order', data.order_id, 'in row', row);\n";
        $script .= "}\n\n";

        $script .= "function addNewOrderRow(sheet, data, headers) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  const rowData = [];\n";
        $script .= "  \n";
        $script .= "  headers.forEach(header => {\n";
        $script .= "    let found = false;\n";
        $script .= "    for (const key in fieldMap) {\n";
        $script .= "      if (fieldMap[key] === header) {\n";
        $script .= "        rowData.push(data[key] || '');\n";
        $script .= "        found = true;\n";
        $script .= "        break;\n";
        $script .= "      }\n";
        $script .= "    }\n";
        $script .= "    if (!found && header !== 'Last Updated') {\n";
        $script .= "      rowData.push('');\n";
        $script .= "    }\n";
        $script .= "  });\n";
        $script .= "  \n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    const now = new Date();\n";
        $script .= "    const formattedDate = Utilities.formatDate(now, Session.getScriptTimeZone(), 'yyyy-MM-dd HH:mm:ss');\n";
        $script .= "    rowData[lastUpdatedIndex] = formattedDate;\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  sheet.appendRow(rowData);\n";
        $script .= "  const newRow = sheet.getLastRow();\n";
        $script .= "  \n";
        $script .= "  formatDateColumns(sheet, headers);\n";
        $script .= "  \n";
        $script .= "  console.log('Added new order', data.order_id, 'in row', newRow);\n";
        $script .= "  return newRow;\n";
        $script .= "}\n\n";

        $script .= "function doPost(e) {\n";
        $script .= "  try {\n";
        $script .= "    const data = JSON.parse(e.postData.contents);\n";
        $script .= "    \n";
        $script .= "    // Get or create the daily sheet\n";
        $script .= "    const orderDate = data.order_date || new Date().toISOString();\n";
        $script .= "    const sheetName = getDailySheetName(orderDate);\n";
        $script .= "    const sheet = getOrCreateSheet(sheetName);\n\n";
        
        $script .= "    // Find order ID column index\n";
        $script .= "    const headers = sheet.getRange(1, 1, 1, sheet.getLastColumn()).getValues()[0];\n";
        $script .= "    const orderIdColumn = headers.indexOf('Order ID') + 1;\n";
        $script .= "    \n";
        $script .= "    if (orderIdColumn === 0) {\n";
        $script .= "      throw new Error('Order ID column not found in sheet headers');\n";
        $script .= "    }\n\n";

        $script .= "    // Check if order already exists\n";
        $script .= "    let existingRow = null;\n";
        $script .= "    if (sheet.getLastRow() > 1) {\n";
        $script .= "      const orderIds = sheet.getRange(2, orderIdColumn, sheet.getLastRow() - 1, 1).getValues();\n";
        $script .= "      \n";
        $script .= "      for (let i = 0; i < orderIds.length; i++) {\n";
        $script .= "        const rowOrderId = orderIds[i][0];\n";
        $script .= "        if (rowOrderId && rowOrderId.toString() === data.order_id.toString()) {\n";
        $script .= "          existingRow = i + 2;\n";
        $script .= "          break;\n";
        $script .= "        }\n";
        $script .= "      }\n";
        $script .= "    }\n\n";

        $script .= "    if (existingRow) {\n";
        $script .= "      updateOrderRow(sheet, existingRow, data, headers);\n";
        $script .= "      \n";
        $script .= "      return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "        status: 'success',\n";
        $script .= "        message: 'Order updated in daily sheet: ' + sheetName,\n";
        $script .= "        order_id: data.order_id,\n";
        $script .= "        sheet: sheetName,\n";
        $script .= "        row: existingRow\n";
        $script .= "      })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "    } else {\n";
        $script .= "      const newRow = addNewOrderRow(sheet, data, headers);\n";
        $script .= "      \n";
        $script .= "      return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "        status: 'success',\n";
        $script .= "        message: 'Order added to daily sheet: ' + sheetName,\n";
        $script .= "        order_id: data.order_id,\n";
        $script .= "        sheet: sheetName,\n";
        $script .= "        row: newRow\n";
        $script .= "      })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "    }\n";

        $script .= "  } catch (error) {\n";
        $script .= "    console.error('Error:', error);\n";
        $script .= "    return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "      status: 'error',\n";
        $script .= "      message: error.toString(),\n";
        $script .= "      stack: error.stack\n";
        $script .= "    })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "// Test function\n";
        $script .= "function testDailySheet() {\n";
        $script .= "  const testData = {\n";
        $script .= "    order_id: 99991,\n";
        $script .= "    billing_name: 'Daily Test Customer',\n";
        $script .= "    billing_email: 'daily@example.com',\n";
        $script .= "    order_status: 'processing',\n";
        $script .= "    order_amount_with_currency: '$75.00',\n";
        $script .= "    order_date: new Date().toISOString()\n";
        $script .= "  };\n";
        $script .= "  \n";
        $script .= "  const mockPost = {\n";
        $script .= "    postData: {\n";
        $script .= "      contents: JSON.stringify(testData)\n";
        $script .= "    }\n";
        $script .= "  };\n";
        $script .= "  \n";
        $script .= "  const result = doPost(mockPost);\n";
        $script .= "  console.log('Daily test result:', result.getContent());\n";
        $script .= "}\n\n";

        $script .= "// List all daily sheets\n";
        $script .= "function listDailySheets() {\n";
        $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
        $script .= "  const sheets = ss.getSheets();\n";
        $script .= "  const dailySheets = [];\n";
        $script .= "  \n";
        $script .= "  sheets.forEach(sheet => {\n";
        $script .= "    const sheetName = sheet.getName();\n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "    // Check for custom template pattern\n";
            $script .= "    if (sheetName.includes('{$custom_template}'.replace(/{[^}]+}/g, ''))) {\n";
        } else {
            $script .= "    if (sheetName.startsWith('Daily - ')) {\n";
        }
        $script .= "      dailySheets.push(sheetName);\n";
        $script .= "    }\n";
        $script .= "  });\n";
        $script .= "  \n";
        $script .= "  console.log('Daily sheets:', dailySheets);\n";
        $script .= "  return dailySheets;\n";
        $script .= "}\n\n";

        $script .= "// Cleanup old daily sheets (older than 30 days)\n";
        $script .= "function cleanupOldDailySheets() {\n";
        $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
        $script .= "  const sheets = ss.getSheets();\n";
        $script .= "  const thirtyDaysAgo = new Date();\n";
        $script .= "  thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);\n";
        $script .= "  \n";
        $script .= "  sheets.forEach(sheet => {\n";
        $script .= "    const sheetName = sheet.getName();\n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "    // Parse date from custom template\n";
            $script .= "    const dateMatch = sheetName.match(/(\\d{4})[\\-\\/]?(\\d{2})[\\-\\/]?(\\d{2})/);\n";
        } else {
            $script .= "    if (sheetName.startsWith('Daily - ')) {\n";
            $script .= "      const dateStr = sheetName.replace('Daily - ', '');\n";
            $script .= "      const dateMatch = dateStr.match(/(\\d{4})-(\\d{2})-(\\d{2})/);\n";
        }
        $script .= "    if (dateMatch) {\n";
        $script .= "      const year = parseInt(dateMatch[1]);\n";
        $script .= "      const month = parseInt(dateMatch[2]) - 1;\n";
        $script .= "      const day = parseInt(dateMatch[3]);\n";
        $script .= "      const sheetDate = new Date(year, month, day);\n";
        $script .= "      \n";
        $script .= "      if (sheetDate < thirtyDaysAgo) {\n";
        $script .= "        ss.deleteSheet(sheet);\n";
        $script .= "        console.log('Deleted old daily sheet:', sheetName);\n";
        $script .= "      }\n";
        $script .= "    }\n";
        if (!$custom_sheet_name || empty($custom_template)) {
            $script .= "    }\n";
        }
        $script .= "  });\n";
        $script .= "}\n";

        return $script;
    }


     /**
     * Generate Google Apps Script code for weekly sheets - PRO FEATURE
     */
    private function generate_weekly_sheet_script($selected_fields, $custom_sheet_name = false, $custom_template = '') {
        // Similar to daily but with weekly logic
        $headers = array();
        $field_mapping = array();

        foreach ($selected_fields as $field_key) {
            if (isset($this->available_fields[$field_key])) {
                $headers[] = $this->available_fields[$field_key]['label'];
                $field_mapping[$field_key] = $this->available_fields[$field_key]['label'];
            }
        }

        $headers_js = json_encode($headers, JSON_PRETTY_PRINT);
        $field_mapping_js = json_encode($field_mapping, JSON_PRETTY_PRINT);
        $field_list = esc_js($this->get_field_list($selected_fields));

        $script  = "// Google Apps Script Code for Google Sheets\n";
        $script .= "// Generated by WP Methods WooCommerce to Google Sheets Plugin - PRO VERSION\n";
        $script .= "// Weekly Sheets Mode\n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "// Custom Sheet Name: {$custom_template}\n";
        }
        $script .= "// Fields: {$field_list}\n\n";

        $script .= "function getWeekNumber(date) {\n";
        $script .= "  const firstDayOfYear = new Date(date.getFullYear(), 0, 1);\n";
        $script .= "  const pastDaysOfYear = (date - firstDayOfYear) / 86400000;\n";
        $script .= "  return Math.ceil((pastDaysOfYear + firstDayOfYear.getDay() + 1) / 7);\n";
        $script .= "}\n\n";

        $script .= "function getWeeklySheetName(dateString) {\n";
        $script .= "  try {\n";
        $script .= "    let date;\n";
        $script .= "    if (dateString.includes('T')) {\n";
        $script .= "      date = new Date(dateString);\n";
        $script .= "    } else if (dateString.includes('-')) {\n";
        $script .= "      const parts = dateString.split(' ')[0].split('-');\n";
        $script .= "      date = new Date(parts[0], parts[1] - 1, parts[2] || 1);\n";
        $script .= "    } else {\n";
        $script .= "      date = new Date();\n";
        $script .= "    }\n";
        $script .= "    \n";
        $script .= "    const year = date.getFullYear();\n";
        $script .= "    const week = getWeekNumber(date);\n";
        $script .= "    \n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "    // Use custom template\n";
            $script .= "    let sheetName = '{$custom_template}';\n";
            $script .= "    sheetName = sheetName.replace('{year}', year);\n";
            $script .= "    sheetName = sheetName.replace('{week}', 'Week ' + week);\n";
            $script .= "    sheetName = sheetName.replace('{site_name}', 'Weekly Orders');\n";
            $script .= "    return sheetName;\n";
        } else {
            $script .= "    return `Weekly - \${year}-W\${String(week).padStart(2, '0')}`;\n";
        }
        $script .= "  } catch (e) {\n";
        $script .= "    const now = new Date();\n";
        $script .= "    const year = now.getFullYear();\n";
        $script .= "    const week = getWeekNumber(now);\n";
        $script .= "    return `Weekly - \${year}-W\${String(week).padStart(2, '0')}`;\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        // Add helper to create or get sheet (same as daily generator)
        $script .= "function getOrCreateSheet(sheetName) {\n";
        $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
        $script .= "  let sheet = ss.getSheetByName(sheetName);\n\n";
        $script .= "  if (!sheet) {\n";
        $script .= "    sheet = ss.insertSheet(sheetName);\n";
        $script .= "    const headers = {$headers_js};\n";
        $script .= "    // Add 'Last Updated' column from the start\n";
        $script .= "    headers.push('Last Updated');\n";
        $script .= "    sheet.appendRow(headers);\n";
        $script .= "    const headerRange = sheet.getRange(1, 1, 1, headers.length);\n";
        $script .= "    headerRange.setBackground('#4CAF50').setFontColor('white').setFontWeight('bold');\n";
        $script .= "    // Format the Last Updated header\n";
        $script .= "    sheet.getRange(1, headers.length).setBackground('#FFC107').setFontWeight('bold');\n";
        $script .= "    sheet.setFrozenRows(1);\n";
        $script .= "    console.log('Created new weekly sheet:', sheetName);\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  return sheet;\n";
        $script .= "}\n\n";

        $script .= "function doPost(e) {\n";
        $script .= "  try {\n";
        $script .= "    const data = JSON.parse(e.postData.contents);\n";

        // Get sheet based on custom naming or weekly name
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "    const sheetName = generateCustomSheetName(data);\n";
            $script .= "    const sheet = getOrCreateSheet(sheetName);\n";
        } else {
            $script .= "    const sheetName = getWeeklySheetName(data.order_date ? data.order_date : new Date().toISOString());\n";
            $script .= "    const sheet = getOrCreateSheet(sheetName);\n";
        }
        
        $script .= "    \n";
        $script .= "    // Initialize sheet if needed with 'Last Updated' column\n";
        $script .= "    if (sheet.getLastRow() === 0) {\n";
        $script .= "      const headers = {$headers_js};\n";
        $script .= "      // Add 'Last Updated' column from the start\n";
        $script .= "      headers.push('Last Updated');\n";
        $script .= "      sheet.appendRow(headers);\n";
        $script .= "      const headerRange = sheet.getRange(1, 1, 1, headers.length);\n";
        $script .= "      headerRange.setBackground('#4CAF50').setFontColor('white').setFontWeight('bold');\n";
        $script .= "      // Format the Last Updated header\n";
        $script .= "      sheet.getRange(1, headers.length).setBackground('#FFC107').setFontWeight('bold');\n";
        $script .= "      sheet.setFrozenRows(1);\n";
        $script .= "      console.log('Sheet initialized with headers:', headers);\n";
        $script .= "    }\n\n";


        $script .= "// Function to format date columns\n";
        $script .= "function formatDateColumns(sheet, headers) {\n";
        $script .= "  const orderDateIndex = headers.indexOf('Order Date');\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  \n";
        $script .= "  if (orderDateIndex !== -1) {\n";
        $script .= "    const orderDateColumn = orderDateIndex + 1;\n";
        $script .= "    const range = sheet.getRange(2, orderDateColumn, sheet.getLastRow() - 1, 1);\n";
        $script .= "    // Set number format to date/time\n";
        $script .= "    range.setNumberFormat('yyyy-mm-dd hh:mm:ss');\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    const lastUpdatedColumn = lastUpdatedIndex + 1;\n";
        $script .= "    const range = sheet.getRange(2, lastUpdatedColumn, sheet.getLastRow() - 1, 1);\n";
        $script .= "    // Set number format to date/time\n";
        $script .= "    range.setNumberFormat('yyyy-mm-dd hh:mm:ss');\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "    // Find order ID column index\n";
        $script .= "    const headers = sheet.getRange(1, 1, 1, sheet.getLastColumn()).getValues()[0];\n";
        $script .= "    const orderIdColumn = headers.indexOf('Order ID') + 1;\n";
        $script .= "    \n";
        $script .= "    if (orderIdColumn === 0) {\n";
        $script .= "      throw new Error('Order ID column not found in sheet headers');\n";
        $script .= "    }\n\n";

        $script .= "    // Check if order already exists\n";
        $script .= "    let existingRow = null;\n";
        $script .= "    if (sheet.getLastRow() > 1) {\n";
        $script .= "      const orderIds = sheet.getRange(2, orderIdColumn, sheet.getLastRow() - 1, 1).getValues();\n";
        $script .= "      \n";
        $script .= "      for (let i = 0; i < orderIds.length; i++) {\n";
        $script .= "        const rowOrderId = orderIds[i][0];\n";
        $script .= "        if (rowOrderId && rowOrderId.toString() === data.order_id.toString()) {\n";
        $script .= "          existingRow = i + 2; // +2 because headers and 0-based index\n";
        $script .= "          break;\n";
        $script .= "        }\n";
        $script .= "      }\n";
        $script .= "    }\n\n";

        $script .= "    if (existingRow) {\n";
        $script .= "      // Update existing order\n";
        $script .= "      updateOrderRow(sheet, existingRow, data, headers);\n";
        $script .= "      \n";
        $script .= "      return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "        status: 'success',\n";
        $script .= "        message: 'Order updated successfully',\n";
        $script .= "        order_id: data.order_id,\n";
        $script .= "        row: existingRow\n";
        $script .= "      })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "    } else {\n";
        $script .= "      // Add new order\n";
        $script .= "      const newRow = addNewOrderRow(sheet, data, headers);\n";
        $script .= "      \n";
        $script .= "      return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "        status: 'success',\n";
        $script .= "        message: 'Order added successfully',\n";
        $script .= "        order_id: data.order_id,\n";
        $script .= "        row: newRow\n";
        $script .= "      })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "    }\n";

        $script .= "  } catch (error) {\n";
        $script .= "    console.error('Error:', error);\n";
        $script .= "    return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "      status: 'error',\n";
        $script .= "      message: error.toString(),\n";
        $script .= "      stack: error.stack\n";
        $script .= "    })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "function updateOrderRow(sheet, row, data, headers) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  \n";
        $script .= "  // Update each field that exists in the data\n";
        $script .= "  for (const key in data) {\n";
        $script .= "    if (fieldMap[key]) {\n";
        $script .= "      // Find which column this field is in\n";
        $script .= "      const columnIndex = headers.indexOf(fieldMap[key]) + 1;\n";
        $script .= "      if (columnIndex > 0) {\n";
        $script .= "        sheet.getRange(row, columnIndex).setValue(data[key]);\n";
        $script .= "      }\n";
        $script .= "    }\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  // Find 'Last Updated' column\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    // Set the timestamp in existing 'Last Updated' column\n";
        $script .= "    sheet.getRange(row, lastUpdatedIndex + 1).setValue(new Date());\n";
        $script .= "  } else {\n";
        $script .= "    console.error('Last Updated column not found in headers:', headers);\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  console.log('Updated order', data.order_id, 'in row', row);\n";
        $script .= "}\n\n";

        $script .= "function addNewOrderRow(sheet, data, headers) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  const rowData = [];\n";
        $script .= "  \n";
        $script .= "  // Build row in the correct order based on headers\n";
        $script .= "  headers.forEach(header => {\n";
        $script .= "    let found = false;\n";
        $script .= "    for (const key in fieldMap) {\n";
        $script .= "      if (fieldMap[key] === header) {\n";
        $script .= "        rowData.push(data[key] || '');\n";
        $script .= "        found = true;\n";
        $script .= "        break;\n";
        $script .= "      }\n";
        $script .= "    }\n";
        $script .= "    if (!found && header !== 'Last Updated') {\n";
        $script .= "      rowData.push('');\n";
        $script .= "    }\n";
        $script .= "  });\n";
        $script .= "  \n";
        $script .= "  // Add timestamp for Last Updated column\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    // Insert timestamp at the correct position\n";
        $script .= "    rowData[lastUpdatedIndex] = new Date();\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  // Add the row\n";
        $script .= "  sheet.appendRow(rowData);\n";
        $script .= "  const newRow = sheet.getLastRow();\n";
        $script .= "  \n";
        $script .= "  // Format date columns\n";
        $script .= "  formatDateColumns(sheet, headers);\n";
        $script .= "  \n";
        $script .= "  console.log('Added new order', data.order_id, 'in row', newRow);\n";
        $script .= "  return newRow;\n";
        $script .= "}\n\n";

        $script .= "function getFieldKeyFromLabel(label) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  for (const key in fieldMap) {\n";
        $script .= "    if (fieldMap[key] === label) return key;\n";
        $script .= "  }\n";
        $script .= "  return label.toLowerCase().replace(/ /g, '_');\n";
        $script .= "}\n\n";

        $script .= "// Test function - run this manually to see if the script works\n";
        $script .= "function testUpdate() {\n";
        $script .= "  const testData = {\n";
        $script .= "    order_id: 12345,\n";
        $script .= "    billing_name: 'Test Customer',\n";
        $script .= "    billing_email: 'test@example.com',\n";
        $script .= "    order_status: 'processing',\n";
        $script .= "    order_amount_with_currency: '$100.00',\n";
        $script .= "    order_date: '2024-01-15 10:30:00'\n";
        $script .= "  };\n";
        $script .= "  \n";
        $script .= "  // Simulate a POST request\n";
        $script .= "  const mockPost = {\n";
        $script .= "    postData: {\n";
        $script .= "      contents: JSON.stringify(testData)\n";
        $script .= "    }\n";
        $script .= "  };\n";
        $script .= "  \n";
        $script .= "  const result = doPost(mockPost);\n";
        $script .= "  console.log('Test result:', result.getContent());\n";
        $script .= "}\n\n";

        $script .= "// Force update all orders - useful if you need to re-sync\n";
        $script .= "function forceInitialize() {\n";
        $script .= "  const sheet = SpreadsheetApp.getActiveSpreadsheet().getActiveSheet();\n";
        $script .= "  sheet.clear();\n";
        $script .= "  \n";
        $script .= "  const headers = {$headers_js};\n";
        $script .= "  // Add 'Last Updated' column from the start\n";
        $script .= "  headers.push('Last Updated');\n";
        $script .= "  sheet.appendRow(headers);\n";
        $script .= "  const headerRange = sheet.getRange(1, 1, 1, headers.length);\n";
        $script .= "  headerRange.setBackground('#4CAF50').setFontColor('white').setFontWeight('bold');\n";
        $script .= "  // Format the Last Updated header\n";
        $script .= "  sheet.getRange(1, headers.length).setBackground('#FFC107').setFontWeight('bold');\n";
        $script .= "  sheet.setFrozenRows(1);\n";
        $script .= "  \n";
        $script .= "  console.log('Sheet initialized with headers:', headers);\n";
        $script .= "}\n";
        
        return $script;
    }


    /**
     * Generate Google Apps Script code for product sheets - PRO FEATURE
     */
    private function generate_product_sheet_script($selected_fields, $custom_sheet_name = false, $custom_template = '') {
        $headers = array();
        $field_mapping = array();

        foreach ($selected_fields as $field_key) {
            if (isset($this->available_fields[$field_key])) {
                $headers[] = $this->available_fields[$field_key]['label'];
                $field_mapping[$field_key] = $this->available_fields[$field_key]['label'];
            }
        }

        $headers_js = json_encode($headers, JSON_PRETTY_PRINT);
        $field_mapping_js = json_encode($field_mapping, JSON_PRETTY_PRINT);
        $field_list = esc_js($this->get_field_list($selected_fields));

        $script  = "// Google Apps Script Code for Google Sheets\n";
        $script .= "// Generated by WP Methods WooCommerce to Google Sheets Plugin - PRO VERSION\n";
        $script .= "// Product Category Sheets Mode\n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "// Custom Sheet Name: {$custom_template}\n";
        }
        $script .= "// Fields: {$field_list}\n\n";

        // Helper functions
        $script .= "function getOrCreateSheet(sheetName) {\n";
        $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
        $script .= "  let sheet = ss.getSheetByName(sheetName);\n\n";
        $script .= "  if (!sheet) {\n";
        $script .= "    sheet = ss.insertSheet(sheetName);\n";
        $script .= "    const headers = {$headers_js};\n";
        $script .= "    // Add 'Last Updated' column from the start\n";
        $script .= "    headers.push('Last Updated');\n";
        $script .= "    sheet.appendRow(headers);\n";
        $script .= "    const headerRange = sheet.getRange(1, 1, 1, headers.length);\n";
        $script .= "    headerRange.setBackground('#4CAF50').setFontColor('white').setFontWeight('bold');\n";
        $script .= "    // Format the Last Updated header\n";
        $script .= "    sheet.getRange(1, headers.length).setBackground('#FFC107').setFontWeight('bold');\n";
        $script .= "    sheet.setFrozenRows(1);\n";
        $script .= "    console.log('Created new product category sheet:', sheetName);\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  return sheet;\n";
        $script .= "}\n\n";

        $script .= "function formatDateColumns(sheet, headers) {\n";
        $script .= "  const orderDateIndex = headers.indexOf('Order Date');\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  \n";
        $script .= "  if (orderDateIndex !== -1) {\n";
        $script .= "    const orderDateColumn = orderDateIndex + 1;\n";
        $script .= "    const range = sheet.getRange(2, orderDateColumn, sheet.getLastRow() - 1, 1);\n";
        $script .= "    range.setNumberFormat('yyyy-mm-dd hh:mm:ss');\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    const lastUpdatedColumn = lastUpdatedIndex + 1;\n";
        $script .= "    const range = sheet.getRange(2, lastUpdatedColumn, sheet.getLastRow() - 1, 1);\n";
        $script .= "    range.setNumberFormat('yyyy-mm-dd hh:mm:ss');\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "function getProductCategorySheetName(data) {\n";
        $script .= "  try {\n";
        $script .= "    // Extract categories from order data\n";
        $script .= "    const categories = data.product_categories || '';\n";
        $script .= "    \n";
        $script .= "    if (!categories) {\n";
        $script .= "      return 'Uncategorized Products';\n";
        $script .= "    }\n";
        $script .= "    \n";
        $script .= "    // Normalize categories: support array or comma-separated string and pick first non-empty\n";
        $script .= "    let categoryList = [];\n";
        $script .= "    if (Array.isArray(categories)) {\n";
        $script .= "      categoryList = categories.map(c => String(c).trim()).filter(Boolean);\n";
        $script .= "    } else {\n";
        $script .= "      categoryList = String(categories).split(',').map(c => c.trim()).filter(Boolean);\n";
        $script .= "    }\n";
        $script .= "    if (categoryList.length === 0) {\n";
        $script .= "      return 'Products - Uncategorized';\n";
        $script .= "    }\n";
        $script .= "    const firstCategory = categoryList[0];\n";
        $script .= "    // Clean up category name for sheet name (remove invalid characters)\n";
        $script .= "    let cleanCategory = firstCategory.replace(/[\\\\/:*?\"<>|]/g, '');\n";
        $script .= "    cleanCategory = cleanCategory.substring(0, 30); // Limit length\n";
        $script .= "    \n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $site_name_js = esc_js(get_bloginfo('name'));
            $script .= "    // Use custom template\n";
            $script .= "    let sheetName = '{$custom_template}';\n";
            $script .= "    sheetName = sheetName.replace(/\{category\}/g, cleanCategory);\n";
            $script .= "    sheetName = sheetName.replace(/\{site_name\}/g, '{$site_name_js}');\n";
            $script .= "    sheetName = sheetName.replace(/[\\\\/:*?\"<>|]/g, '');\n";
            $script .= "    return sheetName;\n";
        } else {
            $site_name_js = esc_js(get_bloginfo('name'));
            $script .= "    return `Products - \${cleanCategory}`;\n";
        }
        $script .= "  } catch (e) {\n";
        $script .= "    console.error('Error getting category:', e);\n";
        if ($custom_sheet_name && !empty($custom_template)) {
            $script .= "    let sheetName = '{$custom_template}';\n";
            $script .= "    sheetName = sheetName.replace(/{category}/g, 'Uncategorized');\n";
            $script .= "    sheetName = sheetName.replace(/{site_name}/g, 'Product Orders');\n";
            $script .= "    sheetName = sheetName.replace(/[\\\\\\/:*?\"<>|]/g, '');\n";
            $script .= "    return sheetName;\n";
        } else {
            $script .= "    return 'Products - Uncategorized';\n";
        }
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "function updateOrderRow(sheet, row, data, headers) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  \n";
        $script .= "  // Update each field that exists in the data\n";
        $script .= "  for (const key in data) {\n";
        $script .= "    if (fieldMap[key]) {\n";
        $script .= "      const columnIndex = headers.indexOf(fieldMap[key]) + 1;\n";
        $script .= "      if (columnIndex > 0) {\n";
            $script .= "        sheet.getRange(row, columnIndex).setValue(data[key]);\n";
            $script .= "      }\n";
            $script .= "    }\n";
            $script .= "  }\n";
            $script .= "  \n";
            $script .= "  // Find 'Last Updated' column\n";
            $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
            $script .= "  if (lastUpdatedIndex !== -1) {\n";
            $script .= "    const now = new Date();\n";
            $script .= "    const formattedDate = Utilities.formatDate(now, Session.getScriptTimeZone(), 'yyyy-MM-dd HH:mm:ss');\n";
            $script .= "    sheet.getRange(row, lastUpdatedIndex + 1).setValue(formattedDate);\n";
            $script .= "  }\n";
            $script .= "  \n";
            $script .= "  formatDateColumns(sheet, headers);\n";
            $script .= "  \n";
            $script .= "  console.log('Updated order', data.order_id, 'in row', row);\n";
            $script .= "}\n\n";

            $script .= "function addNewOrderRow(sheet, data, headers) {\n";
            $script .= "  const fieldMap = {$field_mapping_js};\n";
            $script .= "  const rowData = [];\n";
            $script .= "  \n";
            $script .= "  headers.forEach(header => {\n";
            $script .= "    let found = false;\n";
            $script .= "    for (const key in fieldMap) {\n";
            $script .= "      if (fieldMap[key] === header) {\n";
            $script .= "        rowData.push(data[key] || '');\n";
            $script .= "        found = true;\n";
            $script .= "        break;\n";
            $script .= "      }\n";
            $script .= "    }\n";
            $script .= "    if (!found && header !== 'Last Updated') {\n";
            $script .= "      rowData.push('');\n";
            $script .= "    }\n";
            $script .= "  });\n";
            $script .= "  \n";
            $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
            $script .= "  if (lastUpdatedIndex !== -1) {\n";
            $script .= "    const now = new Date();\n";
            $script .= "    const formattedDate = Utilities.formatDate(now, Session.getScriptTimeZone(), 'yyyy-MM-dd HH:mm:ss');\n";
            $script .= "    rowData[lastUpdatedIndex] = formattedDate;\n";
            $script .= "  }\n";
            $script .= "  \n";
            $script .= "  sheet.appendRow(rowData);\n";
            $script .= "  const newRow = sheet.getLastRow();\n";
            $script .= "  \n";
            $script .= "  formatDateColumns(sheet, headers);\n";
            $script .= "  \n";
            $script .= "  console.log('Added new order', data.order_id, 'in row', newRow);\n";
            $script .= "  return newRow;\n";
            $script .= "}\n\n";

            $script .= "function doPost(e) {\n";
            $script .= "  try {\n";
            $script .= "    const data = JSON.parse(e.postData.contents);\n";
            $script .= "    \n";
            $script .= "    // Get or create the product category sheet\n";
            $script .= "    const sheetName = getProductCategorySheetName(data);\n";
            $script .= "    const sheet = getOrCreateSheet(sheetName);\n\n";
            
            $script .= "    // Find order ID column index\n";
            $script .= "    const headers = sheet.getRange(1, 1, 1, sheet.getLastColumn()).getValues()[0];\n";
            $script .= "    const orderIdColumn = headers.indexOf('Order ID') + 1;\n";
            $script .= "    \n";
            $script .= "    if (orderIdColumn === 0) {\n";
            $script .= "      throw new Error('Order ID column not found in sheet headers');\n";
            $script .= "    }\n\n";

            $script .= "    // Check if order already exists\n";
            $script .= "    let existingRow = null;\n";
            $script .= "    if (sheet.getLastRow() > 1) {\n";
            $script .= "      const orderIds = sheet.getRange(2, orderIdColumn, sheet.getLastRow() - 1, 1).getValues();\n";
            $script .= "      \n";
            $script .= "      for (let i = 0; i < orderIds.length; i++) {\n";
            $script .= "        const rowOrderId = orderIds[i][0];\n";
            $script .= "        if (rowOrderId && rowOrderId.toString() === data.order_id.toString()) {\n";
            $script .= "          existingRow = i + 2;\n";
            $script .= "          break;\n";
            $script .= "        }\n";
            $script .= "      }\n";
            $script .= "    }\n\n";

            $script .= "    if (existingRow) {\n";
            $script .= "      updateOrderRow(sheet, existingRow, data, headers);\n";
            $script .= "      \n";
            $script .= "      return ContentService.createTextOutput(JSON.stringify({\n";
            $script .= "        status: 'success',\n";
            $script .= "        message: 'Order updated in product category sheet: ' + sheetName,\n";
            $script .= "        order_id: data.order_id,\n";
            $script .= "        sheet: sheetName,\n";
            $script .= "        row: existingRow\n";
            $script .= "      })).setMimeType(ContentService.MimeType.JSON);\n";
            $script .= "    } else {\n";
            $script .= "      const newRow = addNewOrderRow(sheet, data, headers);\n";
            $script .= "      \n";
            $script .= "      return ContentService.createTextOutput(JSON.stringify({\n";
            $script .= "        status: 'success',\n";
            $script .= "        message: 'Order added to product category sheet: ' + sheetName,\n";
            $script .= "        order_id: data.order_id,\n";
            $script .= "        sheet: sheetName,\n";
            $script .= "        row: newRow\n";
            $script .= "      })).setMimeType(ContentService.MimeType.JSON);\n";
            $script .= "    }\n";

            $script .= "  } catch (error) {\n";
            $script .= "    console.error('Error:', error);\n";
            $script .= "    return ContentService.createTextOutput(JSON.stringify({\n";
            $script .= "      status: 'error',\n";
            $script .= "      message: error.toString(),\n";
            $script .= "      stack: error.stack\n";
            $script .= "    })).setMimeType(ContentService.MimeType.JSON);\n";
            $script .= "  }\n";
            $script .= "}\n\n";

            $script .= "// Test function for product category sheets\n";
            $script .= "function testProductSheet() {\n";
            $script .= "  const testData = {\n";
            $script .= "    order_id: 99992,\n";
            $script .= "    billing_name: 'Product Category Test Customer',\n";
            $script .= "    billing_email: 'product@example.com',\n";
            $script .= "    order_status: 'processing',\n";
            $script .= "    order_amount_with_currency: '$89.99',\n";
            $script .= "    order_date: new Date().toISOString(),\n";
            $script .= "    product_categories: 'Books, Electronics, Clothing'\n";
            $script .= "  };\n";
            $script .= "  \n";
            $script .= "  const mockPost = {\n";
            $script .= "    postData: {\n";
            $script .= "      contents: JSON.stringify(testData)\n";
            $script .= "    }\n";
            $script .= "  };\n";
            $script .= "  \n";
            $script .= "  const result = doPost(mockPost);\n";
            $script .= "  console.log('Product category test result:', result.getContent());\n";
            $script .= "}\n\n";

            $script .= "// List all product category sheets\n";
            $script .= "function listProductSheets() {\n";
            $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
            $script .= "  const sheets = ss.getSheets();\n";
            $script .= "  const productSheets = [];\n";
            $script .= "  \n";
            $script .= "  sheets.forEach(sheet => {\n";
            $script .= "    const sheetName = sheet.getName();\n";
            if ($custom_sheet_name && !empty($custom_template)) {
                $script .= "    // Check for custom template pattern\n";
                $script .= "    const baseTemplate = '{$custom_template}';\n";
                $script .= "    const templateWithoutPlaceholders = baseTemplate.replace(/{[^}]+}/g, '');\n";
                $script .= "    if (sheetName.includes(templateWithoutPlaceholders)) {\n";
            } else {
                $script .= "    if (sheetName.startsWith('Products - ')) {\n";
            }
            $script .= "      productSheets.push({\n";
            $script .= "        name: sheetName,\n";
            $script .= "        orderCount: Math.max(0, sheet.getLastRow() - 1)\n";
            $script .= "      });\n";
            $script .= "    }\n";
            $script .= "  });\n";
            $script .= "  \n";
            $script .= "  console.log('Product category sheets:', productSheets);\n";
            $script .= "  return productSheets;\n";
            $script .= "}\n\n";

            $script .= "// Function to handle multiple categories per order\n";
            $script .= "function handleMultipleCategories(orderData) {\n";
            $script .= "  const categories = orderData.product_categories || '';\n";
            $script .= "  if (!categories) {\n";
            $script .= "    // Handle in uncategorized sheet\n";
            $script .= "    const sheetName = getProductCategorySheetName(orderData);\n";
            $script .= "    const sheet = getOrCreateSheet(sheetName);\n";
            $script .= "    return [sheet];\n";
            $script .= "  }\n";
            $script .= "  \n";
            $script .= "  const categoryList = categories.split(',').map(cat => cat.trim());\n";
            $script .= "  const sheets = [];\n";
            $script .= "  \n";
            $script .= "  // For each category, create or get sheet and add order\n";
            $script .= "  categoryList.forEach(category => {\n";
            $script .= "    const modifiedData = {...orderData};\n";
            $script .= "    modifiedData.product_categories = category;\n";
            $script .= "    const sheetName = getProductCategorySheetName(modifiedData);\n";
            $script .= "    const sheet = getOrCreateSheet(sheetName);\n";
            $script .= "    sheets.push(sheet);\n";
            $script .= "    \n";
            $script .= "    // Check if order already exists in this category sheet\n";
            $script .= "    const headers = sheet.getRange(1, 1, 1, sheet.getLastColumn()).getValues()[0];\n";
            $script .= "    const orderIdColumn = headers.indexOf('Order ID') + 1;\n";
            $script .= "    let existingRow = null;\n";
            $script .= "    \n";
            $script .= "    if (sheet.getLastRow() > 1) {\n";
            $script .= "      const orderIds = sheet.getRange(2, orderIdColumn, sheet.getLastRow() - 1, 1).getValues();\n";
            $script .= "      for (let i = 0; i < orderIds.length; i++) {\n";
            $script .= "        if (orderIds[i][0] && orderIds[i][0].toString() === orderData.order_id.toString()) {\n";
            $script .= "          existingRow = i + 2;\n";
            $script .= "          break;\n";
            $script .= "        }\n";
            $script .= "      }\n";
            $script .= "    }\n";
            $script .= "    \n";
            $script .= "    if (existingRow) {\n";
            $script .= "      updateOrderRow(sheet, existingRow, orderData, headers);\n";
            $script .= "      console.log(`Updated order in category sheet: \${sheetName}`);\n";
            $script .= "    } else {\n";
            $script .= "      addNewOrderRow(sheet, orderData, headers);\n";
            $script .= "      console.log(`Added order to category sheet: \${sheetName}`);\n";
            $script .= "    }\n";
            $script .= "  });\n";
            $script .= "  \n";
            $script .= "  return sheets;\n";
            $script .= "}\n\n";

            $script .= "// Function to distribute order to all relevant category sheets (advanced mode)\n";
            $script .= "function distributeToAllCategorySheets() {\n";
            $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
            $script .= "  const masterSheet = ss.getSheetByName('All Orders');\n";
            $script .= "  \n";
            $script .= "  if (!masterSheet || masterSheet.getLastRow() <= 1) {\n";
            $script .= "    console.log('No master sheet or no data found');\n";
            $script .= "    return;\n";
            $script .= "  }\n";
            $script .= "  \n";
            $script .= "  const headers = masterSheet.getRange(1, 1, 1, masterSheet.getLastColumn()).getValues()[0];\n";
            $script .= "  const orderIdIndex = headers.indexOf('Order ID');\n";
            $script .= "  const categoriesIndex = headers.indexOf('Product Categories');\n";
            $script .= "  \n";
            $script .= "  if (orderIdIndex === -1 || categoriesIndex === -1) {\n";
            $script .= "    console.log('Required columns not found in master sheet');\n";
            $script .= "    return;\n";
            $script .= "  }\n";
            $script .= "  \n";
            $script .= "  const data = masterSheet.getRange(2, 1, masterSheet.getLastRow() - 1, headers.length).getValues();\n";
            $script .= "  \n";
            $script .= "  data.forEach((row, index) => {\n";
            $script .= "    const orderData = {};\n";
            $script .= "    headers.forEach((header, colIndex) => {\n";
            $script .= "      orderData[header.toLowerCase().replace(/ /g, '_')] = row[colIndex];\n";
            $script .= "    });\n";
            $script .= "    \n";
            $script .= "    orderData.order_id = row[orderIdIndex];\n";
            $script .= "    orderData.product_categories = row[categoriesIndex] || '';\n";
            $script .= "    \n";
            $script .= "    handleMultipleCategories(orderData);\n";
            $script .= "    console.log(`Processed order \${orderData.order_id}`);\n";
            $script .= "  });\n";
            $script .= "  \n";
            $script .= "  console.log('Distribution completed');\n";
            $script .= "}\n\n";

            $script .= "// Function to consolidate all product sheets into one master sheet\n";
            $script .= "function consolidateProductSheets() {\n";
            $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
            $script .= "  const masterSheetName = 'All Product Orders';\n";
            $script .= "  let masterSheet = ss.getSheetByName(masterSheetName);\n";
            $script .= "  \n";
            $script .= "  if (!masterSheet) {\n";
            $script .= "    masterSheet = ss.insertSheet(masterSheetName);\n";
            $script .= "    const headers = {$headers_js};\n";
            $script .= "    headers.push('Last Updated');\n";
            $script .= "    headers.push('Source Sheet');\n";
            $script .= "    masterSheet.appendRow(headers);\n";
            $script .= "    const headerRange = masterSheet.getRange(1, 1, 1, headers.length);\n";
            $script .= "    headerRange.setBackground('#4CAF50').setFontColor('white').setFontWeight('bold');\n";
            $script .= "    masterSheet.setFrozenRows(1);\n";
            $script .= "  }\n";
            $script .= "  \n";
            $script .= "  const productSheets = listProductSheets();\n";
            $script .= "  \n";
            $script .= "  productSheets.forEach(productSheetInfo => {\n";
            $script .= "    const sheet = ss.getSheetByName(productSheetInfo.name);\n";
            $script .= "    if (sheet && sheet.getLastRow() > 1) {\n";
            $script .= "      const data = sheet.getRange(2, 1, sheet.getLastRow() - 1, sheet.getLastColumn()).getValues();\n";
            $script .= "      \n";
            $script .= "      data.forEach(row => {\n";
            $script .= "        const newRow = [...row, productSheetInfo.name];\n";
            $script .= "        masterSheet.appendRow(newRow);\n";
            $script .= "      });\n";
            $script .= "      \n";
            $script .= "      console.log(`Consolidated \${data.length} orders from \${productSheetInfo.name}`);\n";
            $script .= "    }\n";
            $script .= "  });\n";
            $script .= "  \n";
            $script .= "  console.log('Consolidation completed');\n";
            $script .= "}\n";

            return $script;
    }
    
    
    /**
     * Generate Google Apps Script code for monthly sheets - FIXED VERSION
     */
    private function generate_monthly_sheet_script($selected_fields) {
        // Get field labels for headers
        $headers = array();
        $field_mapping = array();

        foreach ($selected_fields as $field_key) {
            if (isset($this->available_fields[$field_key])) {
                $headers[] = $this->available_fields[$field_key]['label'];
                $field_mapping[$field_key] = $this->available_fields[$field_key]['label'];
            }
        }

        $headers_js       = json_encode($headers, JSON_PRETTY_PRINT);
        $field_mapping_js = json_encode($field_mapping, JSON_PRETTY_PRINT);
        $field_list       = esc_js($this->get_field_list($selected_fields));

        $script  = "// Google Apps Script Code for Google Sheets\n";
        $script .= "// Generated by WP Methods WooCommerce to Google Sheets Plugin\n";
        $script .= "// Monthly Sheets Mode - FIXED: No duplicate timestamp columns\n";
        $script .= "// Fields: {$field_list}\n\n";

        $script .= "function doPost(e) {\n";
        $script .= "  try {\n";
        $script .= "    const data = JSON.parse(e.postData.contents);\n";
        $script .= "    \n";
        $script .= "    // Get or create the monthly sheet\n";
        $script .= "    const orderDate = data.order_date || new Date().toISOString();\n";
        $script .= "    const monthYear = getMonthYearFromDate(orderDate);\n";
        $script .= "    const sheet = getOrCreateMonthlySheet(monthYear);\n\n";

        $script .= "    // Find order ID column index\n";
        $script .= "    const headers = sheet.getRange(1, 1, 1, sheet.getLastColumn()).getValues()[0];\n";
        $script .= "    const orderIdColumn = headers.indexOf('Order ID') + 1;\n";
        $script .= "    \n";
        $script .= "    if (orderIdColumn === 0) {\n";
        $script .= "      throw new Error('Order ID column not found in sheet headers');\n";
        $script .= "    }\n\n";

        $script .= "    // Check if order already exists\n";
        $script .= "    let existingRow = null;\n";
        $script .= "    if (sheet.getLastRow() > 1) {\n";
        $script .= "      const orderIds = sheet.getRange(2, orderIdColumn, sheet.getLastRow() - 1, 1).getValues();\n";
        $script .= "      \n";
        $script .= "      for (let i = 0; i < orderIds.length; i++) {\n";
        $script .= "        const rowOrderId = orderIds[i][0];\n";
        $script .= "        if (rowOrderId && rowOrderId.toString() === data.order_id.toString()) {\n";
        $script .= "          existingRow = i + 2; // +2 because headers and 0-based index\n";
        $script .= "          break;\n";
        $script .= "        }\n";
        $script .= "      }\n";
        $script .= "    }\n\n";

        $script .= "    if (existingRow) {\n";
        $script .= "      // Update existing order\n";
        $script .= "      updateOrderRow(sheet, existingRow, data, headers);\n";
        $script .= "      \n";
        $script .= "      return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "        status: 'success',\n";
        $script .= "        message: 'Order updated in ' + monthYear + ' sheet',\n";
        $script .= "        order_id: data.order_id,\n";
        $script .= "        sheet: monthYear,\n";
        $script .= "        row: existingRow\n";
        $script .= "      })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "    } else {\n";
        $script .= "      // Add new order\n";
        $script .= "      const newRow = addNewOrderRow(sheet, data, headers);\n";
        $script .= "      \n";
        $script .= "      return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "        status: 'success',\n";
        $script .= "        message: 'Order added to ' + monthYear + ' sheet',\n";
        $script .= "        order_id: data.order_id,\n";
        $script .= "        sheet: monthYear,\n";
        $script .= "        row: newRow\n";
        $script .= "      })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "    }\n";

        $script .= "  } catch (error) {\n";
        $script .= "    console.error('Error:', error);\n";
        $script .= "    return ContentService.createTextOutput(JSON.stringify({\n";
        $script .= "      status: 'error',\n";
        $script .= "      message: error.toString(),\n";
        $script .= "      stack: error.stack\n";
        $script .= "    })).setMimeType(ContentService.MimeType.JSON);\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "function getMonthYearFromDate(dateString) {\n";
        $script .= "  try {\n";
        $script .= "    // Handle different date formats\n";
        $script .= "    let date;\n";
        $script .= "    if (dateString.includes('T')) {\n";
        $script .= "      // ISO format\n";
        $script .= "      date = new Date(dateString);\n";
        $script .= "    } else if (dateString.includes('-')) {\n";
        $script .= "      // YYYY-MM-DD format\n";
        $script .= "      const parts = dateString.split(' ')[0].split('-');\n";
        $script .= "      date = new Date(parts[0], parts[1] - 1, parts[2] || 1);\n";
        $script .= "    } else {\n";
        $script .= "      // Use current date as fallback\n";
        $script .= "      date = new Date();\n";
        $script .= "    }\n";
        $script .= "    \n";
        $script .= "    const monthNames = ['January', 'February', 'March', 'April', 'May', 'June', \n";
        $script .= "                       'July', 'August', 'September', 'October', 'November', 'December'];\n";
        $script .= "    const month = monthNames[date.getMonth()];\n";
        $script .= "    const year = date.getFullYear();\n";
        $script .= "    \n";
        $script .= "    return month + ' ' + year;\n";
        $script .= "  } catch (e) {\n";
        $script .= "    // If date parsing fails, use current month\n";
        $script .= "    const now = new Date();\n";
        $script .= "    const monthNames = ['January', 'February', 'March', 'April', 'May', 'June', \n";
        $script .= "                       'July', 'August', 'September', 'October', 'November', 'December'];\n";
        $script .= "    return monthNames[now.getMonth()] + ' ' + now.getFullYear();\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "function getOrCreateMonthlySheet(monthYear) {\n";
        $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
        $script .= "  let sheet = ss.getSheetByName(monthYear);\n\n";
        $script .= "  if (!sheet) {\n";
        $script .= "    sheet = ss.insertSheet(monthYear);\n";
        $script .= "    const headers = {$headers_js};\n";
        $script .= "    // Add 'Last Updated' to headers from the start\n";
        $script .= "    headers.push('Last Updated');\n";
        $script .= "    sheet.appendRow(headers);\n";
        $script .= "    const headerRange = sheet.getRange(1, 1, 1, headers.length);\n";
        $script .= "    headerRange.setBackground('#4CAF50').setFontColor('white').setFontWeight('bold');\n";
        $script .= "    // Format the Last Updated header\n";
        $script .= "    sheet.getRange(1, headers.length).setBackground('#FFC107').setFontWeight('bold');\n";
        $script .= "    sheet.setFrozenRows(1);\n";
        $script .= "    \n";
        $script .= "    console.log('Created new sheet:', monthYear);\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  return sheet;\n";
        $script .= "}\n\n";

        $script .= "function updateOrderRow(sheet, row, data, headers) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  \n";
        $script .= "  // Update each field that exists in the data\n";
        $script .= "  for (const key in data) {\n";
        $script .= "    if (fieldMap[key]) {\n";
        $script .= "      // Find which column this field is in\n";
        $script .= "      const columnIndex = headers.indexOf(fieldMap[key]) + 1;\n";
        $script .= "      if (columnIndex > 0) {\n";
        $script .= "        sheet.getRange(row, columnIndex).setValue(data[key]);\n";
        $script .= "      }\n";
        $script .= "    }\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  // Find 'Last Updated' column\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    // Set the timestamp in existing 'Last Updated' column with proper formatting\n";
        $script .= "    const now = new Date();\n";
        $script .= "    // Format as YYYY-MM-DD HH:MM:SS to match Order Date format\n";
        $script .= "    const formattedDate = Utilities.formatDate(now, Session.getScriptTimeZone(), 'yyyy-MM-dd HH:mm:ss');\n";
        $script .= "    sheet.getRange(row, lastUpdatedIndex + 1).setValue(formattedDate);\n";
        $script .= "  } else {\n";
        $script .= "    console.error('Last Updated column not found in headers:', headers);\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  // Format date columns\n";
        $script .= "  formatDateColumns(sheet, headers);\n";
        $script .= "  \n";
        $script .= "  console.log('Updated order', data.order_id, 'in row', row);\n";
        $script .= "}\n\n";

        $script .= "function addNewOrderRow(sheet, data, headers) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  const rowData = [];\n";
        $script .= "  \n";
        $script .= "  // Build row in the correct order based on headers\n";
        $script .= "  headers.forEach(header => {\n";
        $script .= "    let found = false;\n";
        $script .= "    for (const key in fieldMap) {\n";
        $script .= "      if (fieldMap[key] === header) {\n";
        $script .= "        rowData.push(data[key] || '');\n";
        $script .= "        found = true;\n";
        $script .= "        break;\n";
        $script .= "      }\n";
        $script .= "    }\n";
        $script .= "    if (!found && header !== 'Last Updated') {\n";
        $script .= "      rowData.push('');\n";
        $script .= "    }\n";
        $script .= "  });\n";
        $script .= "  \n";
        $script .= "  // Add timestamp for Last Updated column with proper formatting\n";
        $script .= "  const lastUpdatedIndex = headers.indexOf('Last Updated');\n";
        $script .= "  if (lastUpdatedIndex !== -1) {\n";
        $script .= "    const now = new Date();\n";
        $script .= "    // Format as YYYY-MM-DD HH:MM:SS to match Order Date format\n";
        $script .= "    const formattedDate = Utilities.formatDate(now, Session.getScriptTimeZone(), 'yyyy-MM-dd HH:mm:ss');\n";
        $script .= "    // Insert timestamp at the correct position\n";
        $script .= "    rowData[lastUpdatedIndex] = formattedDate;\n";
        $script .= "  }\n";
        $script .= "  \n";
        $script .= "  // Add the row\n";
        $script .= "  sheet.appendRow(rowData);\n";
        $script .= "  const newRow = sheet.getLastRow();\n";
        $script .= "  \n";
        $script .= "  console.log('Added new order', data.order_id, 'in row', newRow);\n";
        $script .= "  return newRow;\n";
        $script .= "}\n";

        $script .= "function getFieldKeyFromLabel(label) {\n";
        $script .= "  const fieldMap = {$field_mapping_js};\n";
        $script .= "  for (const key in fieldMap) {\n";
        $script .= "    if (fieldMap[key] === label) return key;\n";
        $script .= "  }\n";
        $script .= "  return label.toLowerCase().replace(/ /g, '_');\n";
        $script .= "}\n\n";

        $script .= "// Debug function to see all monthly sheets\n";
        $script .= "function listMonthlySheets() {\n";
        $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
        $script .= "  const sheets = ss.getSheets();\n";
        $script .= "  const monthlySheets = [];\n";
        $script .= "  \n";
        $script .= "  sheets.forEach(sheet => {\n";
        $script .= "    const sheetName = sheet.getName();\n";
        $script .= "    // Check if sheet name matches monthly pattern\n";
        $script .= "    if (sheetName.match(/^(January|February|March|April|May|June|July|August|September|October|November|December) \\d{4}$/)) {\n";
        $script .= "      monthlySheets.push(sheetName);\n";
        $script .= "    }\n";
        $script .= "  });\n";
        $script .= "  \n";
        $script .= "  console.log('Monthly sheets:', monthlySheets);\n";
        $script .= "  return monthlySheets;\n";
        $script .= "}\n\n";

        $script .= "// Manual test function\n";
        $script .= "function testMonthlyUpdate() {\n";
        $script .= "  const testData = {\n";
        $script .= "    order_id: 99999,\n";
        $script .= "    billing_name: 'Monthly Test Customer',\n";
        $script .= "    billing_email: 'monthly@example.com',\n";
        $script .= "    order_status: 'processing',\n";
        $script .= "    order_amount_with_currency: '$150.00',\n";
        $script .= "    order_date: new Date().toISOString()\n";
        $script .= "  };\n";
        $script .= "  \n";
        $script .= "  // Simulate a POST request\n";
        $script .= "  const mockPost = {\n";
        $script .= "    postData: {\n";
        $script .= "      contents: JSON.stringify(testData)\n";
        $script .= "    }\n";
        $script .= "  };\n";
        $script .= "  \n";
        $script .= "  const result = doPost(mockPost);\n";
        $script .= "  console.log('Monthly test result:', result.getContent());\n";
        $script .= "}\n";

        return $script;
    }


    /**
     * Get custom sheet name functions
     */
    private function get_custom_sheet_name_functions($template) {
        $site_name_js = esc_js(get_bloginfo('name'));

        $script = "function generateCustomSheetName(data) {\n";
        $script .= "  try {\n";
        $script .= "    let sheetName = '{$template}';\n";
        $script .= "    // Determine base date: prefer order date if provided\n";
        $script .= "    let baseDate = null;\n";
        $script .= "    try {\n";
        $script .= "      if (data && data.order_date) {\n";
        $script .= "        const dstr = String(data.order_date);\n";
        $script .= "        if (dstr.indexOf('T') !== -1) {\n";
        $script .= "          baseDate = new Date(dstr);\n";
        $script .= "        } else if (dstr.indexOf('-') !== -1) {\n";
        $script .= "          const parts = dstr.split(' ')[0].split('-');\n";
        $script .= "          baseDate = new Date(parts[0], (parts[1] ? parts[1]-1 : 0), (parts[2] || 1));\n";
        $script .= "        } else {\n";
        $script .= "          baseDate = new Date(dstr);\n";
        $script .= "        }\n";
        $script .= "      }\n";
        $script .= "    } catch (e) { baseDate = null; }\n";
        $script .= "    const now = baseDate || new Date();\n";
        $script .= "    const monthNames = ['January','February','March','April','May','June','July','August','September','October','November','December'];\n";
        $script .= "    function getWeekNumber(d) {\n";
        $script .= "      d = new Date(Date.UTC(d.getFullYear(), d.getMonth(), d.getDate()));\n";
        $script .= "      const dayNum = d.getUTCDay() || 7;\n";
        $script .= "      d.setUTCDate(d.getUTCDate() + 4 - dayNum);\n";
        $script .= "      const yearStart = new Date(Date.UTC(d.getUTCFullYear(),0,1));\n";
        $script .= "      return Math.ceil((((d - yearStart) / 86400000) + 1)/7);\n";
        $script .= "    }\n";

        $script .= "    // Replace variables (based on order date when present)\n";
        $script .= "    sheetName = sheetName.replace(/\{month\}/g, monthNames[now.getMonth()]);\n";
        $script .= "    sheetName = sheetName.replace(/\{year\}/g, now.getFullYear());\n";
        $script .= "    sheetName = sheetName.replace(/\{day\}/g, String(now.getDate()).padStart(2,'0'));\n";
        $script .= "    sheetName = sheetName.replace(/\{week\}/g, 'Week ' + getWeekNumber(now));\n";
        $script .= "    sheetName = sheetName.replace(/\{site_name\}/g, '{$site_name_js}');\n";

        $script .= "    // Add order count if needed\n";
        $script .= "    if (/\{order_count\}/.test(sheetName)) {\n";
        $script .= "      const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
        $script .= "      let totalOrders = 0;\n";
        $script .= "      const sheets = ss.getSheets();\n";
        $script .= "      sheets.forEach(sh => { totalOrders += Math.max(0, sh.getLastRow() - 1); });\n";
        $script .= "      sheetName = sheetName.replace(/\{order_count\}/g, String(totalOrders + 1));\n";
        $script .= "    }\n";

        $script .= "    // Final cleanup\n";
        $script .= "    sheetName = sheetName.replace(/[\\\\/:*?\"<>|]/g, '');\n";
        $script .= "    return sheetName;\n";
        $script .= "  } catch (e) {\n";
        $script .= "    return '{$site_name_js} - Orders';\n";
        $script .= "  }\n";
        $script .= "}\n\n";

        $script .= "function getOrCreateSheet(sheetName) {\n";
        $script .= "  const ss = SpreadsheetApp.getActiveSpreadsheet();\n";
        $script .= "  let sheet = ss.getSheetByName(sheetName);\n\n";
        $script .= "  if (!sheet) {\n";
        $script .= "    sheet = ss.insertSheet(sheetName);\n";
        $script .= "    console.log('Created new sheet:', sheetName);\n";
        $script .= "  }\n";
        $script .= "  return sheet;\n";
        $script .= "}\n\n";
        
        return $script;
    }
    
    /**
     * Get field list string
     */
    private function get_field_list($selected_fields) {
        $field_names = array();
        foreach ($selected_fields as $field_key) {
            if (isset($this->available_fields[$field_key])) {
                $field_names[] = $this->available_fields[$field_key]['label'];
            }
        }
        return implode(', ', $field_names);
    }


    /**
     * Generate Google Apps Script code for single sheet - UPDATED to use UGSIW_Script_Generator
     */
    private function generate_google_apps_script_single($selected_fields) {
        // Pull pro feature settings so generated script matches admin settings
        $daily_weekly = get_option('ugsiw_gs_daily_weekly', 'none');
        $product_sheets = get_option('ugsiw_gs_product_sheets', '0');
        $custom_sheet_name = get_option('ugsiw_gs_custom_sheet_name', '0');
        $custom_template = get_option('ugsiw_gs_custom_name_template', 'Orders - {month} {year}');

        $pro_features = array(
            'daily_weekly'     => $daily_weekly,
            'product_sheets'   => ($product_sheets === '1' || $product_sheets === 1 || $product_sheets === true),
            'custom_sheet_name'=> ($custom_sheet_name === '1' || $custom_sheet_name === 1 || $custom_sheet_name === true),
            'custom_template'  => $custom_template,
        );

        return $this->generate_script($selected_fields, false, $pro_features);
    }

    /**
     * Generate Google Apps Script code for monthly sheets - UPDATED to use UGSIW_Script_Generator
     */
    private function generate_google_apps_script_monthly($selected_fields) {
        // Pull pro feature settings so generated script matches admin settings
        $daily_weekly = get_option('ugsiw_gs_daily_weekly', 'none');
        $product_sheets = get_option('ugsiw_gs_product_sheets', '0');
        $custom_sheet_name = get_option('ugsiw_gs_custom_sheet_name', '0');
        $custom_template = get_option('ugsiw_gs_custom_name_template', 'Orders - {month} {year}');

        $pro_features = array(
            'daily_weekly'     => $daily_weekly,
            'product_sheets'   => ($product_sheets === '1' || $product_sheets === 1 || $product_sheets === true),
            'custom_sheet_name'=> ($custom_sheet_name === '1' || $custom_sheet_name === 1 || $custom_sheet_name === true),
            'custom_template'  => $custom_template,
        );

        return $this->generate_script($selected_fields, true, $pro_features);
    }
}